'use client';

import { motion } from 'framer-motion';
import Image from 'next/image';
import { useState } from 'react';

export default function RsvpForm() {
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        location: '',
        availability: ''
    });
    const [isLoading, setIsLoading] = useState(false);
    const [status, setStatus] = useState<{
        type: 'success' | 'error' | null;
        message: string;
    }>({ type: null, message: '' });

    const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
        setFormData(prev => ({
            ...prev,
            [e.target.name]: e.target.value
        }));
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsLoading(true);
        setStatus({ type: null, message: '' });

        try {
            const response = await fetch('/api/send-rsvp', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(formData),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || 'Something went wrong');
            }

            setStatus({
                type: 'success',
                message: 'Thank you for your RSVP! We look forward to celebrating with you.'
            });
            setFormData({ name: '', email: '', location: '', availability: '' });
        } catch (error) {
            setStatus({
                type: 'error',
                message: error instanceof Error ? error.message : 'Failed to submit RSVP'
            });
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <div className="container mx-auto px-4 py-20">
            <motion.div
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.8 }}
                className="max-w-2xl mx-auto text-center space-y-8"
            >
                <h2 className="text-coral font-script text-4xl md:text-5xl">RSVP</h2>

                <div className="space-y-4 text-[1.25rem] leading-8">
                    <p className="font-normal">Strictly by invitation only | No Children allowed</p>
                    <p className="font-light">
                        We&apos;re so grateful for your support and we would love to host everyone, but with
                        budget constraints, we had to make some tough decisions to keep it intimate.
                    </p>
                    <p className="font-light">
                        Kindly RSVP on/before 28th February, 2025
                    </p>
                </div>

                <div className="relative">
                    {/* Decorative leaves - left */}
                    <div className="absolute -left-[200px] top-0 -z-1 hidden md:block">
                        <Image
                            src="/leaves.png"
                            alt="Decorative leaves"
                            width={220}
                            height={80}
                        />
                    </div>

                    {/* Form */}
                    <form onSubmit={handleSubmit} className="bg-transparent p-8 md:p-10 rounded-lg relative border border-[#222222] z-10">
                        {status.type && (
                            <div className={`mb-6 p-4 rounded ${status.type === 'success' ? 'bg-mint/20 text-green-800' : 'bg-red-100 text-red-800'}`}>
                                {status.message}
                            </div>
                        )}

                        <div className="space-y-6">
                            <input
                                type="text"
                                name="name"
                                value={formData.name}
                                onChange={handleChange}
                                placeholder="Name"
                                className="w-full bg-transparent border-b border-[#222]/20 py-2 px-1 focus:outline-none focus:border-coral"
                                required
                            />

                            <input
                                type="text"
                                name="location"
                                value={formData.location}
                                onChange={handleChange}
                                placeholder="Where are you coming from"
                                className="w-full bg-transparent border-b border-[#222]/20 py-2 px-1 focus:outline-none focus:border-coral"
                                required
                            />

                            <input
                                type="email"
                                name="email"
                                value={formData.email}
                                onChange={handleChange}
                                placeholder="E-mail"
                                className="w-full bg-transparent border-b border-[#222]/20 py-2 px-1 focus:outline-none focus:border-coral"
                                required
                            />

                            <select
                                name="availability"
                                value={formData.availability}
                                onChange={handleChange}
                                className="w-full bg-transparent border-b border-[#222]/20 py-2 px-1 focus:outline-none focus:border-coral"
                                required
                            >
                                <option value="">Select your availability</option>
                                <option value="available" className="">I&apos;ll be there!</option>
                                <option value="virtual" className="">I&apos;ll join virtually</option>
                                <option value="not_available" className="">Sorry, can&apos;t make it</option>
                                <option value="uncertain" className="">Not sure yet</option>
                            </select>
                        </div>

                        <button
                            type="submit"
                            disabled={isLoading}
                            className="btn-outline-dark text-sm mt-10 md:mt-12 disabled:opacity-50"
                        >
                            {isLoading ? 'PROCESSING...' : 'REGISTER'}
                        </button>
                    </form>

                    {/* Decorative leaves - right */}
                    <div className="absolute -right-[200px] top-0 -z-1 scale-x-[-1] hidden md:block">
                        <Image
                            src="/leaves.png"
                            alt="Decorative leaves"
                            width={220}
                            height={80}
                        />
                    </div>
                </div>
            </motion.div>
        </div>
    );
} 