"use client"

import { useState, useEffect } from 'react'
import Link from "next/link";
import Image from "next/image";

import MenuIcon from "./icons/menu";
import CloseIcon from "./icons/close";

const leftMenuItems = [
    { label: "home", link: "//" },
    { label: "couple", link: "//#couple" },
    { label: "story", link: "//#story" },
]

const rightMenuItems = [
    { label: "gift", link: "//#gift" },
    { label: "gallery", link: "//#gallery" },
    { label: "blog", link: "/blog" },
]

const Navbar = ({ topPlacement = false }: { topPlacement?: boolean }) => {

    const [isScrolled, setIsScrolled] = useState(false);
    const [prevScrollPos, setPrevScrollPos] = useState(0);
    const [visible, setVisible] = useState(true);
    const [activeSection, setActiveSection] = useState('home');
    const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);

    useEffect(() => {
        let rafId: number;
        const handleScroll = () => {
            const currentScrollPos = window.scrollY;

            rafId = requestAnimationFrame(() => {
                if (currentScrollPos < 50) {
                    setIsScrolled(false);
                    setVisible(true);
                } else {
                    setIsScrolled(true);
                    setVisible((prevScrollPos > currentScrollPos) || currentScrollPos < 100);
                }

                setPrevScrollPos(currentScrollPos);

                // Track current section
                const currentPath = window.location.pathname;
                if (currentPath === '/blog' || currentPath.includes('/blog/')) {
                    setActiveSection('blog');
                } else {
                    const sections = ['home', 'couple', 'story', 'rsvp', 'gift', 'gallery'];
                    for (const section of sections) {
                        const element = document.getElementById(section);
                        if (element) {
                            const rect = element.getBoundingClientRect();
                            if (rect.top <= 150 && rect.bottom >= 150) {
                                setActiveSection(section);
                                break;
                            }
                        }
                    }
                }
            });
        };

        window.addEventListener('scroll', handleScroll, { passive: true });

        return () => {
            window.removeEventListener('scroll', handleScroll);
            if (rafId) {
                cancelAnimationFrame(rafId);
            }
        };
    }, [prevScrollPos]);

    const allMenuItems = [...leftMenuItems, ...rightMenuItems];
    return (
        <nav
            className={`w-full transition-all duration-300 ease-in-out fixed ${isScrolled || topPlacement
                ? 'top-0 bg-white/90 backdrop-blur-sm'
                : 'md:bottom-0 md:top-auto top-0 bg-white/90'
                } ${!visible ? '-translate-y-full' : 'translate-y-0'
                } z-50 border-b border-[#222222]/10`}
        >
            <div className="container md:max-w-[1200px] mx-auto px-4">
                {/* Desktop Navigation */}
                <div className="hidden md:flex justify-between items-center h-20">
                    <div className="flex-1 flex justify-between items-center space-x-12">
                        {leftMenuItems.map((menuItem) => (<Link
                            href={menuItem.link}
                            key={menuItem.link}
                            className={`block-letters text-sm  hover:text-[#222222] ${activeSection === menuItem.label ? "text-[#222222]" : "text-[#222222]/40"
                                }`}
                        >
                            {menuItem.label}
                        </Link>))}
                        <Link href="/">
                            <div className="w-24 relative">
                                <Image
                                    src="/ABC-Logo.png"
                                    alt="Love is ABC Logo"
                                    width={60}
                                    height={60}
                                    className="object-contain"
                                />
                            </div>
                        </Link>
                        {rightMenuItems.map((menuItem) => (<Link
                            href={menuItem.link}
                            key={menuItem.link}
                            className={`block-letters text-sm hover:text-[#222222] ${activeSection === menuItem.label ? "text-[#222222]" : "text-[#222222]/40"
                                }`}
                        >
                            {menuItem.label}
                        </Link>))}
                    </div>
                </div>

                {/* Mobile Navigation */}
                <div className="md:hidden flex justify-between items-center h-16">
                    <Link href="/">
                        <div className="w-16 relative">
                            <Image
                                src="/ABC-Logo.png"
                                alt="Love is ABC Logo"
                                width={40}
                                height={40}
                                className="object-contain"
                            />
                        </div>
                    </Link>
                    <button
                        onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
                        className="p-2"
                    >
                        <MenuIcon className="w-6 h-6 text-[#222222]" />
                    </button>
                </div>
            </div>

            {/* Mobile Menu Overlay */}
            {isMobileMenuOpen && (
                <div className="fixed top-0 left-0 w-full h-screen bg-white z-50 md:hidden">
                    <div className="flex flex-col items-center justify-start h-full space-y-8">
                        <div className="relative w-full">
                            <button
                                onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
                                className="p-2 absolute top-[.8rem] right-2"
                            >
                                <CloseIcon className="w-6 h-6 text-[#222222]" />
                            </button>
                        </div>
                        <div className="w-20 relative mb-8">
                            <Image
                                src="/ABC-Logo.png"
                                alt="Love is ABC Logo"
                                width={80}
                                height={80}
                                className="object-contain"
                            />
                        </div>
                        {allMenuItems.map((menuItem) => (
                            <Link
                                href={menuItem.link}
                                key={menuItem.link}
                                onClick={() => setIsMobileMenuOpen(false)}
                                className={`block-letters text-md text-[#222222]/40 hover:text-[#222222] ${activeSection === menuItem.label && "text-[#222222]"
                                    }`}
                            >
                                {menuItem.label}
                            </Link>
                        ))}
                    </div>
                </div>
            )}
        </nav>
    )
}

export default Navbar